/*global define */
/*jslint white: true */

/*
	TimeItem:

	This object provides a mixin for each time based item.
*/

define(["src/utils"],
function(utils) {
	'use strict';

	var kMaxTime = 36000; // 10 hours!

	function TimeItem() {
		utils.assert(this !== undefined, "TimeItem not called with new.");
		// do not modify these private members directly or notifiers won't be called correctly
		this.pTimeData = { 
			parentOffsetTime	: 0,				//"raw" (untrimmed) start time, relative to parent's time system
			durationTime		: kMaxTime,			//duration of "raw" (untrimmed) item, starting at untrimmed start
			trimInTime			: 0,				//time trimmed off the start; parentOffset + this = visible start time
			trimOutTime			: kMaxTime,			//out time trimmed relative to the beginning; parentOffset + this = visible end time
			minSampleTime		: undefined,		//minimum time we can sample the item from
			maxSampleTime		: undefined			//maximum time we can sample the item from
		};
	}

	utils.mixin(TimeItem, {

		clone : function (clone_children, other) {
			var result = other;

			if (result) {
				// init
				TimeItem.call(result);
			} else {
				// alloc and init
				result = new TimeItem();
			}

			utils.clone(false, result.pTimeData, this.pTimeData);

			return result;
		},

		// Get trimmed start time, relative to the parent's time system
		getParentIn : function () {
			return this.pTimeData.parentOffsetTime + this.pTimeData.trimInTime;
		},
		
		// Get trimmed end time, relative to the parent's time system
		getParentOut : function () {
			return this.pTimeData.parentOffsetTime + this.pTimeData.trimOutTime;
		},
		
		getDuration : function () {
			return this.pTimeData.durationTime;
		},
		
		getMaxTime : function () {
			return kMaxTime;
		},
		
		// Get "raw" (untrimmed) start time, relative to the parent's time system
		getParentOffsetTime : function () {
			return this.pTimeData.parentOffsetTime;
		},
		
		getTrimInTime : function () {
			return this.pTimeData.trimInTime;
		},
		
		getTrimOutTime : function () {
			return this.pTimeData.trimOutTime;
		},
		
		setParentOffsetTime : function (t) {
			utils.assert(t !== undefined && t !== true && t !== false, "setParentOffsetTime invalid t parameter");
			this.pTimeData.parentOffsetTime = t;
		},
		
		setDuration : function (t) {
			utils.assert(t !== undefined && t !== true && t !== false, "setDuration invalid t parameter");
			this.pTimeData.durationTime = t;
		},
		
		setTrimInTime : function (t) {
			utils.assert(t !== undefined && t !== true && t !== false, "setTrimInTime invalid t parameter");
			this.pTimeData.trimInTime = t;
		},
		
		setTrimOutTime : function (t) {
			utils.assert(t !== undefined && t !== true && t !== false, "setTrimInTime invalid t parameter");
			this.pTimeData.trimOutTime = t;
		},

		// May return undefined
		getMinSampleTime : function () {
			return this.pTimeData.minSampleTime;
		},

		// May return undefined
		getMaxSampleTime : function () {	
			return this.pTimeData.maxSampleTime;
		},

		// May return undefined
		setMinSampleTime : function (t) {
			this.pTimeData.minSampleTime = t;
		},

		// May return undefined
		setMaxSampleTime : function (t) {	
			this.pTimeData.maxSampleTime = t;
		},

		convertToLocalTime : function (inTimeSpan, inParent0) {
			var timeSpan = {t:inTimeSpan.t, dt:inTimeSpan.dt};

			// If we introduce stretching, dt will need to change.
			// Note, the TimeItem object holds the time translation data, so we don't need to use inParent0;
			timeSpan.t -= this.getParentOffsetTime();

			return timeSpan;
		},
	
		isActivateAtTime : function (inLocalTime) {
			return inLocalTime >= this.getTrimInTime() && inLocalTime < this.getTrimOutTime();	
		}
	});

	return TimeItem;
});
